local std = require("lib.std")

local M = {}
M.__index = M

function M.fes(header, footer)
	local config = {}
	local site_config = {}
	local fes_mod = package.loaded.fes
	if fes_mod and fes_mod.config then
		config = fes_mod.config
		if config.site then
			site_config = config.site
		end
	end

	if site_config.favicon then
		site_config.favicon = '<link rel="icon" type="image/x-icon" href="' .. site_config.favicon .. '">'
	end

	local self = {
		version = site_config.version,
		title = site_config.title,
		copyright = site_config.copyright,
		favicon = site_config.favicon,
		header = header or [[
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width,initial-scale=1.0">
{{FAVICON}}
<title>{{TITLE}}</title>
<style>
:root {
  --bg: #f5f5f5;
  --text: #111827;
  --muted: #6b7280;
  --link: #1a0dab;
  --accent: #68a6ff;
  --highlight: #004d99;
  --note-bg: #ffffff;
  --panel-bg: #ffffff;
  --border: rgba(0,0,0,.1);
  --table-head: #f3f4f6;
  --code-color: #004d99;
  --blockquote-border: #1a73e8;
  --banner-bg: #ffffff;
  --footer-bg: #ffffff;
  --shadow: rgba(0,0,0,.08);
}

@media (prefers-color-scheme: dark) {
  :root {
    --bg: #0f1113;
    --text: #e6eef3;
    --muted: #9aa6b1;
    --link: #68a6ff;
    --accent: #68a6ff;
    --highlight: #cde7ff;
    --note-bg: #1a1c20;
    --panel-bg: #1a1c20;
    --border: rgba(255,255,255,.06);
    --table-head: #1a1c20;
    --code-color: #cde7ff;
    --blockquote-border: #68a6ff;
    --banner-bg: #1a1c20;
    --footer-bg: #1a1c20;
    --shadow: rgba(0,0,0,.4);
  }
}

html, body {
  min-height: 100%;
  margin: 0;
  padding: 0;
  background: var(--bg);
  color: var(--text);
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
  line-height: 1.5;
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
}

body { padding: 36px; }

.container { max-width: 830px; margin: 0 auto; }

.container > *:not(.banner) { margin: 28px 0; }

h1, h2, h3, h4, h5, h6 { font-weight: 600; margin: 0 0 12px 0; }

h1 { font-size: 40px; margin-bottom: 20px; font-weight: 700; }

h2 { font-size: 32px; margin: 26px 0 14px; }

h3 { font-size: 26px; margin: 22px 0 12px; }

h4 { font-size: 20px; margin: 18px 0 10px; }

h5 { font-size: 16px; margin: 16px 0 8px; }

h6 { font-size: 14px; margin: 14px 0 6px; color: var(--muted); }

p { margin: 14px 0; }

a { color: var(--link); text-decoration: none; transition: color .15s ease, text-decoration-color .15s ease; }

.hidden { color: var(--text); text-decoration: none; }

a:hover { text-decoration: underline; }

summary { cursor: pointer; }

details {
  background: var(--panel-bg);
  border: 1px solid var(--border);
  border-radius: 4px;
  padding: 14px 16px;
  margin: 16px 0;
}

details summary {
  list-style: none;
  font-weight: 600;
  color: var(--text);
  display: flex;
  align-items: center;
}

details summary::-webkit-details-marker { display: none; }

details summary::before {
  content: "▸";
  margin-right: 8px;
  transition: transform .15s ease;
  color: var(--accent);
}

details[open] summary::before { transform: rotate(90deg); }

summary::after { content: "Expand"; margin-left: auto; font-size: 13px; color: var(--muted); }

details[open] summary::after { content: "Collapse"; }

details > *:not(summary) { margin-top: 12px; }

.note, pre, code {
  background: var(--note-bg);
  border: 1px solid var(--border);
}

.note {
  padding: 20px;
  border-radius: 4px;
  background: var(--note-bg);
  border: 1px solid var(--border);
  margin: 28px 0;
  color: var(--text);
}

.note strong { color: var(--text); }

.muted { color: var(--muted); }

.lead { font-size: 15px; margin-top: 8px; }

.callout { display: block; margin: 12px 0; }

.small { font-size: 13px; color: var(--muted); margin-top: 6px; }

.highlight { font-weight: 700; color: var(--highlight); }

ul, ol { margin: 14px 0; padding-left: 26px; }

.tl {
  display: grid;
  grid-template-columns: repeat(auto-fill, 200px);
  gap: 15px;
  list-style-type: none;
  padding: 0;
  margin: 0;
  justify-content: start;
}

ul.tl li { padding: 10px; width: fit-content; }

li { margin: 6px 0; }

code {
  padding: 3px 7px;
  border-radius: 3px;
  font-family: "SF Mono", Monaco, "Cascadia Code", "Roboto Mono", Consolas, "Courier New", monospace;
  font-size: .9em;
  color: var(--code-color);
}

pre {
  padding: 20px;
  border-radius: 4px;
  margin: 14px 0;
  overflow-x: auto;
  font-family: "SF Mono", Monaco, "Cascadia Code", "Roboto Mono", Consolas, "Courier New", monospace;
  font-size: 14px;
  line-height: 1.6;
}

pre code { background: none; border: none; padding: 0; font-size: inherit; }

blockquote {
  border-left: 3px solid var(--blockquote-border);
  padding-left: 18px;
  margin: 14px 0;
  color: var(--text);
  font-style: italic;
}

hr { border: 0; border-top: 1px solid rgba(0,0,0,.08); margin: 26px 0; }

@media (prefers-color-scheme: dark) {
  hr { border-top-color: rgba(255,255,255,.1); }
}

img { max-width: 100%; height: auto; border-radius: 4px; margin: 14px 0; }

table { width: 100%; border-collapse: collapse; margin: 14px 0; }

th, td {
  padding: 12px 16px;
  text-align: left;
  border-bottom: 1px solid var(--border);
}

th {
  background: var(--table-head);
  font-weight: 600;
  color: var(--text);
}

tr:hover { background: rgba(0,0,0,0.02); }

@media (prefers-color-scheme: dark) {
  tr:hover { background: rgba(255,255,255,0.02); }
}

.divider { margin: 26px 0; height: 1px; background: rgba(0,0,0,.08); }

@media (prefers-color-scheme: dark) {
  .divider { background: rgba(255,255,255,.1); }
}

.section { margin-top: 36px; }

.links { margin: 12px 0; }

.links a { display: inline-block; margin: 0 14px 6px 0; color: var(--link); }

strong, b { font-weight: 600; color: var(--text); }

em, i { font-style: italic; }

.center { display: flex; justify-content: center; align-items: center; }

.banner {
  width: 100%;
  box-sizing: border-box;
  text-align: center;
  background: var(--banner-bg);
  padding: 20px;
  border: 1px solid var(--border);
  border-bottom-right-radius: 8px;
  border-bottom-left-radius: 8px;
  color: var(--text);
  margin: -36px 0 28px 0;
  box-shadow: 0 0.2em 0.6em var(--shadow);
}

.nav { margin-left: auto; margin-right: auto; }

.nav a { color: var(--highlight); }

.footer {
  background: var(--footer-bg);
  padding: 20px 0;
  border-top: 1px solid rgba(0,0,0,.08);
  font-size: 14px;
  color: var(--muted);
  display: flex;
  justify-content: center;
  align-items: center;
  gap: 24px;
  margin-top: 28px !important;
  margin-bottom: 0;
}

.left { text-align: left; float: left; }

.right { text-align: right; float: right; }
</style>
</head>
<body>
<div class="container">
]],
		footer = footer or [[
<footer class="footer">
<a href="https://git.vxserver.dev/fSD/fes" target="_blank">Fes Powered</a>
<a href="https://www.lua.org/" target="_blank">Lua Powered</a>
<a href="https://git.vxserver.dev/fSD/fes/src/branch/master/COPYING" target="_blank">ISC Licensed</a>
<p>{{COPYRIGHT}}</p>
</footer>
</div>
</body>
</html>
]],
		parts = {},
	}

	return setmetatable(self, M)
end

function M:custom(str)
	table.insert(self.parts, str)
	return self
end

for name, func in pairs(std) do
	if type(func) == "function" then
		M[name] = function(self, ...)
			local result = func(...)
			table.insert(self.parts, result)
			return self
		end
	end
end

function M:build()
	local header = self.header
	header = header:gsub("{{TITLE}}", self.title or "Document")
	local favicon_html = self.favicon and ('<link rel="icon" type="image/x-icon" href="' .. self.favicon .. '">')
	header = header:gsub(
		"{{FAVICON}}",
		favicon_html
			or [[<link rel="icon" href="data:image/svg+xml,<svg xmlns=%%22http://www.w3.org/2000/svg%%22 viewBox=%%220 0 100 100%%22><text y=%%22.9em%%22 font-size=%%2290%%22>🔥</text></svg>">]]
	)
	local footer = self.footer:gsub("{{COPYRIGHT}}", self.copyright or "&#169; The Copyright Holder")
	return header .. table.concat(self.parts, "\n") .. footer
end

M.__tostring = function(self)
	return self:build()
end

return M
